// VNX_SWtest.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"


// ---- The dll has __stdcall format, so we need to un-decorate the function call names ----
extern "C" {
#include "VNX_switch.h"

}

#include <string>
#include <algorithm>
using namespace std;


// ---------- Demonstration Program ------------------------------------------------------
//
// This program shows how to call the RF switch functions
//
// It is not intended as an example of appropriate user interface design, or the error handling
// that would be present in a production program. Neither is it an example of elegant C style, or 
// optimum efficiency.
//
// Several examples of testing for error codes returned by DLL function calls are included, but for simplicity
// many example calls do not test for error cases that might be returned.
//
// Note that the example code gets the state of the switch hardware before the commands are sent to it, running
// the program again will display the settings of the device which resulted from the last run of the program.
 

// ------------------------------- Allocations -------------------------------------------

static DEVID MyDevices[MAXDEVICES];			// I have statically allocated this array for convenience
											// It holds a list of device IDs for the connected devices
											// They are stored starting at MyDevices[0]

static char MyDeviceName[MAX_MODELNAME];	// NB -- its a char array for now, no support for wchars, etc.

static bool gbWantOneDevice		= FALSE;
static int DevNum				= 1;		// the device we should work with - index is UI version - 1.


// --------------------------- Variables -------------------------------------------------


static long SerialNumber		= 0;		// used to hold the serial number for the get serial number command
static int Switch_Select		= 0;		// used to hold the switch number to select
static int Set_External_Control = 0;		// used to choose internal or external switch control signals
static int Use_Pulse_Mode		= 0;		// used to enable pulsed mode operation if 1
static int Start_Pattern		= 0;		// used to start the pattern if 1
static int RFOnOff				= 1;		// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode			= 0;		// the low byte is the sweep command byte
static int GetParam				= 0;		// the low byte is the GET command byte

BOOL gbWantSwitch				= FALSE;	// the user wants to set the switch
BOOL gbWantExtControl			= FALSE;	// the user wants to set internal vs external control signals
BOOL gbWantSaveSettings			= FALSE;	// the user wants to save the settings
BOOL gbWantPulseMode			= FALSE;	// the user wants to start or stop pulsed mode
BOOL gbWantGetParam				= FALSE;	// the user wants to get a parameter
BOOL gbWantSwitchPattern		= FALSE;
BOOL gbGotReply					= FALSE;






// -------------------------------- Command Line routines --------------------------------
void PrintHelp()
{
	printf("Vaunix RF Switch Demo\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection. Defaults to first device ---\n");
	printf("  -d nn		Select the device to work with, nn is the device number (1,2,3, etc.)\n");
	printf("  -y		Save the current settings in the device\n");
	printf("\n");

	printf(" --- Commands to set parameters and start patterns or pulse mode --- \n");
	printf("  -s n      Set the switch, n is the active switch number\n");
	printf("  -x n      Select internal (n=0) or external (n=1) control signals\n");
	printf("  -p n      Start pulse mode, 1 = on, 0 to stop\n");
	printf("  -g n      Start a switch pattern, 1 = once, 2 = repeating, 0 to stop\n");

	printf("\n");


}
//.-----------------------------------------------------------------------------------
char MakeLower(char &c)
{
	return c = tolower(c);
}

// -----------------------------------------------------------------------------------
// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should quit for some other reason.
BOOL ParseCommandLine(int argc, char *argv[])
{
	enum { wantDash, wantDevSubstring, wantSwitch, wantExtControl, wantPulseMode, wantSwitchPattern, wantGetParam, wantDevID } state = wantDash;

	for (int i = 1; i < argc; ++i) {
		// Convert each argument to lowercase
		string thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLower);

		BOOL bHasDash = ('-' == thisParam[0]);
		if (bHasDash) {
			if (state != wantDash) {
				PrintHelp();
				return FALSE;
			}

			// remove the dash from the front of the string
			thisParam = string(thisParam.begin() + 1, thisParam.end());

			// Identify the command line arguments
			if ("d" == thisParam) {
				state = wantDevID;
			}
			else if ("x" == thisParam) {
				gbWantExtControl = TRUE;
				state = wantExtControl;
			}
			else if ("s" == thisParam) {
				gbWantSwitch = TRUE;
				state = wantSwitch;
			}
			else if ("p" == thisParam) {
				gbWantPulseMode = TRUE;
				state = wantPulseMode;
			}
			else if ("g" == thisParam) {
				gbWantSwitchPattern = TRUE;
				state = wantSwitchPattern;
			}
			else if ("y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if ("q" == thisParam) {
				gbWantGetParam = TRUE;
				state = wantGetParam;
			}

			else {
				// this case is for "-h" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		} else {


			// save the whole substring and do conversions for each argument type

			switch (state){
			
			case wantSwitch:
				Switch_Select = atoi(thisParam.c_str());		// get the switch number (1,2,3,4)
				state = wantDash;
				break;

			case wantDevID:
				DevNum = atoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantExtControl:
				Set_External_Control = atoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantPulseMode:
				Use_Pulse_Mode = atoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantSwitchPattern:
				Start_Pattern = atoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantGetParam:
				GetParam = atoi(thisParam.c_str());
				state = wantDash;
				break;
			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}

// -------------------------------- Program Main -----------------------------------------

int main(int argc, char* argv[])
{
	int i, j, k;
	int itemp, n_entries;
	bool bTemp;
	float ftemp, PulseOnTime, PulseRepTime;

	printf("Lab Brick RF Switch DLL Demo Program\n");

	if (!ParseCommandLine(argc, argv))
		return 0;

//	 -- convert the user's device number to our internal MyDevices array index --
//	 printf("DevNum after command line parsing = %d\n", DevNum);

	DevNum = DevNum - 1;

	if (DevNum < 0) DevNum = 0;

	if (DevNum > MAXDEVICES - 1) DevNum = MAXDEVICES - 1;

	printf("Device Selected = %d\n", DevNum + 1);

	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLSW_SetTestMode(FALSE);
	
	i = fnLSW_GetNumDevices();

	if (i == 0){
		printf("No device found\n");
	}

	if (i == 1){
		printf("Found %d Device\n", i);

	}
	else {
		printf("Found %d Devices\n", i);
	}

	i = fnLSW_GetDevInfo(MyDevices);

	printf("Got Device Info for %d Device[s]\n", i);

	
	if ( i > 0 )	// do we have a device?
	{
		for (j = 0; j < i; j++){

		// --- print out the first device's name ---
		itemp = fnLSW_GetModelName(MyDevices[j], MyDeviceName);
		printf("Device %d is an %s \n", j+1, MyDeviceName);

		// --- print out the device's serial number ---
		itemp = fnLSW_GetSerialNumber(MyDevices[j]);
		printf("Device %d has serial number %d \n", j+1, itemp);


		// --- We need to init the device (open it) before we can do anything else ---
		itemp = fnLSW_InitDevice(MyDevices[j]);

		if (itemp){
			printf("InitDevice returned %x\n", itemp);
		}
		
		// --- Lets see if we got the device's settings and parameters ---
		
		itemp = fnLSW_GetNumSwitches(MyDevices[j]);
		printf("There are %d switches\n", itemp);

		itemp = fnLSW_GetSwitchSetting(MyDevices[j]);
		printf("Switch %d is selected\n", itemp);

		itemp = fnLSW_GetActiveSwitch(MyDevices[j]);
		printf("Switch %d is active\n", itemp);

		
		// --- We'll see what the switch pattern type was last set to ---

		itemp = fnLSW_GetPatternType(MyDevices[j]);
		if (!itemp){
			printf("No pattern active\n");
		}
		if (itemp & SWPAT_REPEAT){
			printf("Repeating pattern mode\n");
		}
		if (itemp & SWPAT_ONCE){
			printf("Single pattern mode\n");
		}
		// --- and some of the other pattern values ---

		n_entries = fnLSW_GetPatternLength(MyDevices[j]);
		if (n_entries & STATUS_ERROR){
			printf("fnLSW_GetPatternLength returned error code %x\n", n_entries);
		}
		else printf("There are %d entries in the pattern\n", n_entries);

		for (k = 0; k < n_entries; k++)
		{
			itemp = fnLSW_GetPatternEntryTime(MyDevices[j], k);
			printf("Hold time for switch pattern entry %d is %d \n", k, itemp);

			itemp = fnLSW_GetPatternEntrySwitch(MyDevices[j], k);
			printf("Switch setting for switch pattern entry %d is %d \n", k, itemp);

		} // end of for loop over pattern entries


		// --- and print out some other settings from the device ---

		itemp = fnLSW_GetUseExternalControl(MyDevices[j]);
		if (itemp){
			printf("Using external control signals\n");
		}
		else {
			printf("Using internal control signals\n");
		}

		ftemp = fnLSW_GetPulseOnTime(MyDevices[j]);
		printf("Pulse On Time = %g seconds, ", ftemp);

		ftemp = fnLSW_GetPulseOffTime(MyDevices[j]);
		printf("Pulse Off Time = %g seconds\n", ftemp);

		// --- get the operation mode for pulsed operation, showing testing for errors in the return value

		itemp = fnLSW_GetPulseMode(MyDevices[j]);
		if (itemp & STATUS_ERROR){
			printf("fnLSW_GetPulseMode returned error code %x\n", itemp);
		}
		else {
			if (itemp) printf("Pulsed mode on\n");
			else printf("Pulsed mode off\n");
		}

		itemp = fnLSW_GetHasFastPulseMode(MyDevices[j]);
		if (itemp & STATUS_ERROR){
			printf("fnLSW_GetHasFastPulseMode returned error code %x\n", itemp);
		}
		else {
			if (itemp) printf("Pulsed mode available\n");
			else printf("Pulsed mode not available\n");
		}



		} // end of the for loop over the devices


		// if the user is trying to control a device we don't have, then quit now

		if (DevNum > i - 1){

			for (j = 0; j < i; j++)
			{
				itemp = fnLSW_CloseDevice(MyDevices[j]);
			}

			return 0;			// quit - nothing else to do
		}
	


		// ------------- Now we'll set the requested device with new parameters -------------

		printf("Setting the Lab Brick parameters..\n");


		// --- Lets set the switch first ---

		if (gbWantSwitch)
		{
			itemp = fnLSW_SetSwitch(MyDevices[DevNum], Switch_Select);
			if (itemp) printf("fnLSW_SetSwitch returned %x\n", itemp);
		}

		// --- and then do whatever else the user requested ---
		
		if (gbWantExtControl)
		{
			if (Set_External_Control == 0)
			{	bTemp = FALSE;}
			else
			{	bTemp = TRUE;}

			itemp = fnLSW_SetUseExternalControl(MyDevices[DevNum], bTemp);
		}

		if (gbWantPulseMode)
		{
			// set up some arbitrary pulse modulation values and
			PulseOnTime = .001;
			PulseRepTime = .005;
			// start the pulse modulation if the user wants to, otherwise stop pulse modulation
			if (Use_Pulse_Mode)
			{
					itemp = fnLSW_SetFastPulsedOutput(MyDevices[DevNum], PulseOnTime, PulseRepTime, TRUE);
					if (itemp) printf("fnLSW_SetFastPulsedOutput returned %x\n", itemp);
			}
			else 
			{
					itemp = fnLSW_EnableInternalPulseMod(MyDevices[DevNum], FALSE);
					if (itemp) printf("fnLSW_EnableInternalPulseMod returned %x\n", itemp);
			}
		}
	
		if (gbWantSwitchPattern)
		{
			// start a pattern if the user wants to, otherwise stop whatever pattern is active

			if (Start_Pattern == 0)
			{
				printf("Stopping the Switch Pattern\n");
				itemp = fnLSW_StartPattern(MyDevices[DevNum], FALSE);
				if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
			}
			else
			{
			// --- first we'll generate a sample pattern ---
				int Switch_Select[4] = {1,2,1,2};					// the selected switch
				int Switch_HoldTime[4] = {1000,2000,3000,4000};		// hold times for each switch in milliseconds

			// --- Then we'll send the pattern to the hardware ---

			itemp = fnLSW_SetPattern(MyDevices[DevNum], 4, Switch_Select, Switch_HoldTime);
			if (itemp) printf("fnLSW_SetPattern returned %x\n", itemp);

			// --- and then we'll tell the hardware to execute the pattern ---

				if (Start_Pattern == 2)
				{
					itemp = fnLSW_SetPatternType(MyDevices[DevNum], TRUE);	// the user wants a repeating pattern
				}
				else
				{
					itemp = fnLSW_SetPatternType(MyDevices[DevNum], FALSE);	// the user wants the pattern to run once

				}

				printf("Starting the Switch Pattern\n");
				itemp = fnLSW_StartPattern(MyDevices[DevNum], TRUE);
				if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
			}
		}


		if (gbWantSaveSettings)
		{
			fnLSW_SaveSettings(MyDevices[DevNum]);
		}


		
		// -- Lets wait around some and report on the device's operation
		
		j = 0;

		while (j < 40)
		{

			itemp = fnLSW_GetActiveSwitch(MyDevices[DevNum]);
			printf("Switch %d is active\n", itemp); 

			Sleep(500);		// wait for 1/2 second
			j++;
		
		}
		
		// -- we've done whatever the user wanted, time to close the devices
		
		for (j = 0; j < i; j++)
		{
			itemp = fnLSW_CloseDevice(MyDevices[j]);

		}

	} // end of if ( i > 0 ) -- "we have a device"

	return 0;
}

// --------------------------- end of main ----------------------------------
